import numpy as np
import scipy.optimize as spo



"""
Schema d'Euler explicite pour l'equation autonome X'(t) = f(X(t))

- x0 est la condition initiale
- N est le nombre de pas
- T est le temps final
- f est la fonction du systeme autonome X'= f(X(t))

- la fonction retourne l'approximation x[n] de X(tn) aux temps tn = n * h = n  * T / N
"""
def EulerExpl(x0, N, T, f):
    h = T / N
    X = np.empty(N+1)
    X[0] = x0

    # y(n+1) = y(n) + h * f(y(n))
    for i in range(N):
        X[i+1] = X[i] + h * f(X[i])

    return X


"""
Schema d'Euler implicite pour l'equation autonome X'(t) = f(X(t))

- x0 est la condition initiale
- N est le nombre de pas
- T est le temps final
- f est la fonction du systeme autonome

- la fonction retourne l'approximation x[n] de X(tn) aux temps tn = n * h = n  * T / N
"""
def EulerImpl(x0, N, T, f):
    h = T / N 
    X = np.empty(N+1)
    X[0] = x0

    def func(x, xn, h, f):
        return xn + h * f(x) - x
    
    # y(n+1) = y(n) + h * f(y(n+1))
    for i in range(N):
        X[i+1] = spo.fsolve(func, X[i], (X[i], h, f))

    return X


"""
Schema du point milieu explicite (RK2) pour l'equation autonome X'(t) = f(X(t))

- x0 est la condition initiale
- N est le nombre de pas
- T est le temps final
- f est la fonction du systeme autonome

- la fonction retourne l'approximation x[n] de X(tn) aux temps tn = n * h = n  * T / N
"""
def RK2(x0, N, T, f):
    h = T /N
    X = np.empty(N+1)
    X[0] = x0

    # y1 = y(n) + h/2 * f(y(n))
    # y(n+1) = y(n) + h * f(y1)
    for i in range(N):
        X[i+1] = X[i] + h * f(X[i] + 0.5 * h * f(X[i]))

    return X



"""
Schema de Heun pour l'equation autonome X'(t) = f(X(t))

- x0 est la condition initiale
- N est le nombre de pas
- T est le temps final
- f est la fonction du systeme autonome

- la fonction retourne l'approximation x[n] de X(tn) aux temps tn = n * h = n  * T / N
"""
def Heun(x0, N, T, f):
    h = T / N
    X = np.empty(N+1)
    X[0] = x0

    # y1 = y(n)
    # y2 = y(n) + h * f(y(n))
    # y(n+1) = y(n) + h/2 * (f(y1) + f(y2))
    for i in range(N):
        X[i+1] = X[i] + 0.5 * h * (f(X[i]) + f(X[i] + h * f(X[i])))

    return X

"""
Schema RK4 pour l'equation autonome X'(t) = f(X(t))

- x0 est la condition initiale
- N est le nombre de pas
- T est le temps final
- f est la fonction du systeme autonome

- la fonction retourne l'approximation x[n] de X(tn) aux temps tn = n * h = n  * T / N
"""
def RK4(x0, N, T, f):
    h = T / N
    X = np.empty(N+1)
    X[0] = x0

    # k1 = f(y(n))
    # k2 = f(y(n) + h/2 * k1)
    # k3 = f(y(n) + h/2 * k2)
    # k4 = f(y(n) + h * k3)
    # y(n+1) = y(n) = h/6 * (k1 + 2*k2 + 2*k3 + k4)
    for i in range(N):
        k1 = f(X[i])
        k2 = f(X[i] + 0.5 * h * k1)
        k3 = f(X[i] + 0.5 * h * k2)
        k4 = f(X[i] + h * k3)
        X[i+1] = X[i] + h/6 * (k1 + 2*k2 + 2*k3 + k4)

    return X


"""
Schema de Crank-Nicolson pour l'equation autonome X'(t) = f(X(t))

- x0 est la condition initiale
- N est le nombre de pas
- T est le temps final
- f est la fonction du systeme autonome

- la fonction retourne l'approximation x[n] de X(tn) aux temps tn = n * h = n  * T / N
"""
def CN(x0, N, T, f):
    h = T / N 
    X = np.empty(N+1)
    X[0] = x0

    def func(x, xn, h, f):
        return xn +0.5*h*f(xn)+ 0.5*h * f(x) - x
    
    # y(n+1) = y(n) + h * f(y(n+1))
    for i in range(N):
        X[i+1] = spo.fsolve(func, X[i], (X[i], h, f))

    return X